package mcfall.math;

/**
 * A RowVector is a Vector whose values are stored in a single row.  It is therefore an 1xm Matrix
 * @author mcfall
 *
 */
public class RowVector extends Vector {
	/*  
	 * Constructs a row vector having the specified number of columns
	 * @param numColumns a positive integer indicating the number of columns for this RowVector
	 */
	public RowVector (int numColumns) {
		super (1, numColumns);
	}
	
	/*
	 * Constructs a row vector with the specified number of columns and the specified initial values.  If
	 * <i>initialValues</i> has fewer than numColumns values, the first n components of the vector will be
	 * initialized, and the remaining components will be 0.  If initialValues contains more than numColumns
	 * values, the first numRows values will be used to initialize the vector, and the remaining values
	 * will be ignored
	 * @param numColumns a positive integer specifying the number of columns in this RowVector
	 * @param initialValues an array of double values used to initialize the vector components
	 * 
	 */
	public RowVector (int numColumns, double[] initialValues) {
		super (1, numColumns, initialValues);
	}

	/**
	 * Constructs a 4-value Row vector that travels in the direction needed to go from the point
	 * <i>from</i> to the point <i>to</i>
	 * @param from the starting location of the vector
	 * @param to the ending location of the vector
	 */
	public RowVector (Point from, Point to) {
		this(4);
		setValueAt(1, to.getX()-from.getX());
		setValueAt(2, to.getY()-from.getY());
		setValueAt(3, to.getZ()-from.getZ());
	}
	
	@Override
	public double setValueAt(int i, double value) throws IndexOutOfBoundsException {
		return super.setValueAt(getFirstRowIndex(), i, value);
		
	}

	@Override
	public double getValueAt(int i) throws IndexOutOfBoundsException {
		return super.getValueAt(getFirstRowIndex(), i);
	}

	@Override
	public Vector duplicate() {
		RowVector copy = new RowVector (getNumberOfColumns());
		copy.setFirstColumnIndex(getFirstColumnIndex());
		copy.setFirstRowIndex(getFirstRowIndex());
		for (int column = getFirstColumnIndex(); column <= getLastColumnIndex(); column++) {
			copy.setValueAt(column, getValueAt(column));
		}
		return copy;
	}

	@Override
	public int getFirstIndex() {
		return getFirstColumnIndex();
	}

	@Override
	public int getSize() {
		return getNumberOfColumns();
	}
}
